import os, argparse, numpy as np
from pathlib import Path

def main():
    ap = argparse.ArgumentParser()
    ap.add_argument("--src", required=True, help="root of E0.npz tree")
    ap.add_argument("--dst", required=True, help="output root (mirrors src)")
    ap.add_argument("--sigma", type=float, default=3.0, help="LoG sigma for polarity decision")
    args = ap.parse_args()

    try:
        from scipy.ndimage import gaussian_laplace
    except Exception as e:
        raise SystemExit("SciPy required: install with `conda install scipy` (or `pip install scipy`)")

    src, dst = Path(args.src), Path(args.dst)
    count = flips = 0

    for p in src.rglob("E0.npz"):
        rel = p.relative_to(src)
        out = dst / rel
        out.parent.mkdir(parents=True, exist_ok=True)

        E0 = np.load(p)["E0"].astype(np.float64)
        z = (E0 - E0.mean()) / (E0.std() + 1e-9)
        LoG = gaussian_laplace(z, sigma=args.sigma)
        pos = np.maximum(LoG, 0.0).sum(dtype=np.float64)
        neg = np.maximum(-LoG, 0.0).sum(dtype=np.float64)
        flip = bool(neg > pos)

        np.savez_compressed(out, E0=(-E0 if flip else E0))
        count += 1
        flips += int(flip)

    print(f"wrote {count} E0 files into {dst}")
    print(f"flipped {flips} of them to align curvature with +LoG tail")

if __name__ == "__main__":
    main()
